#!/bin/sh

# Helper script to enable, disable, or check status of backups to a remote
# location.

set -e

usage() {
    echo "usage: $1 <backup-config-dir> enable|disable|status|test" >&2
    exit 1
}

if [ $# -ne 2 ]; then
    usage $0
fi

BACKUP_CFG_DIR=$1
shift

BACKUP_VARS=${BACKUP_CFG_DIR}/wg_backup_vars
if [ -r ${BACKUP_VARS} ] ; then
    . ${BACKUP_VARS}
    BACKUP_CONFIGURED=1
else
    BACKUP_CONFIGURED=0
fi

case "$1" in
"enable" )
    [ $BACKUP_CONFIGURED -ne 1 ] && exit 0
    if [ "${WG_BACKUP_METHOD}" = "sftp" ] ; then
        if [ ! -d "${WG_BACKUP_MNT}" ] ; then
            mkdir -p "${WG_BACKUP_MNT}"
            chmod 775 "${WG_BACKUP_MNT}"
        fi
        if /bin/mount | grep -q "${WG_BACKUP_MNT}\>" ; then
            echo "Backup appears to be already enabled." 1>&2
            exit 1
        fi
        /usr/bin/sshfs -o idmap=user,reconnect,nonempty,workaround=rename \
            -F "${WG_BACKUP_SSH_CONFIG}" \
            "${WG_BACKUP_HOST}:${WG_BACKUP_HOST_DIR}" "${WG_BACKUP_MNT}"
    fi
    ;;
"disable")
    [ $BACKUP_CONFIGURED -ne 1 ] && exit 0
    if [ "${WG_BACKUP_METHOD}" = "sftp" ] ; then
        if /bin/mount | grep -q "${WG_BACKUP_MNT}\>" ; then
            /bin/fusermount -z -u "${WG_BACKUP_MNT}"
        fi
    fi
    ;;
"status")
    if [ $BACKUP_CONFIGURED -ne 1 ] ; then
        echo "Backup is not configured."
        exit 0
    fi 
    if [ "${WG_BACKUP_METHOD}" = "sftp" ] ; then
        if /bin/mount | grep -q "${WG_BACKUP_MNT}\>" ; then
            echo "Backup is configured and enabled."
        else
            echo "Backup is configured but currently disabled."
        fi
    else
            echo "Unsupported backup method -- ${WG_BACKUP_METHOD}."
    fi
    ;;
"test")
    if [ $BACKUP_CONFIGURED -ne 1 ] ; then
        echo "Backup is not configured."
        exit 0
    fi
    if [ "${WG_BACKUP_METHOD}" = "sftp" ] ; then
        rc=0
        echo ls | /usr/bin/sftp -b - -F "${WG_BACKUP_SSH_CONFIG}" \
            "${WG_BACKUP_HOST}":"${WG_BACKUP_HOST_DIR}" >/dev/null || rc=$?
        if [ $rc -eq 0 ] ; then
            echo "SSH sftp subsystem test succeeded."
        else
            echo "SSH sftp subsystem test failed." 1>&2
            exit 1
        fi
    else
        echo "Unsupported backup method -- ${WG_BACKUP_METHOD}." 1>&2
        exit 1
    fi
    ;;
*)
    usage $0
    ;;
esac

exit 0
