#!/bin/sh

set -e

update_apt_keyring() {
    if [ -f "/root/trustedkeys.gpg" ] ; then 
        gpg --homedir /root --no-default-keyring --keyring /root/trustedkeys.gpg --export cm@watchguard.com | apt-key add -
    fi
}

usage() {
    echo "usage: $1 install|upgrade|uninstall" >&2
    exit 1
}

if [ $# -ne 1 ] ; then
    usage $0
fi

case "$1" in
    "install" | "upgrade")

        if [ -r /etc/default/wg_system ] ; then
            . /etc/default/wg_system
        else
            # Nothing to do
            exit 0
        fi

        if [ ! -x /sbin/initctl ] ; then
            # Upstart not installed. Nothing to do at the moment.
            exit 0
        fi

        # Check if wg_systemd is already registered and make sure it is not
        # running
        if [ -f /etc/init/wg_systemd.conf ] ; then
            echo "Starting wg_systemd"
            /sbin/initctl stop wg_systemd >/dev/null || true
        fi

        # Do the same for wg_backup
        if [ -f /etc/init/wg_backup.conf ] ; then
            echo "Starting wg_backup"
            /sbin/initctl stop wg_backup >/dev/null || true
        fi

        # Create $WG_ADMIN_GROUP system group
        rc=0
        getent group "$WG_ADMIN_GROUP" >/dev/null || rc=$?
        if [ $rc -eq 2 ] ; then
            echo "Creating $WG_ADMIN_GROUP system group"
            groupadd -r "$WG_ADMIN_GROUP"
        fi

        # Create or modify $WG_SUPPORT_USER system account
        rc=0
        getent passwd "$WG_SUPPORT_USER" >/dev/null || rc=$?
        if [ $rc -eq 2 ] ; then

            # Create $WG_SUPPORT_USER system account with password
            # "readwrite"
            echo "Creating $WG_SUPPORT_USER system account"
            password='$6$xzv66s/J$gIYEf07igw9lAruhGDZq4leDBwe3sy28SAu17KJ78sZoTqJRAEhK/bsHeQiQEa0SZuBWEC9zVXp97rmRGG/cI1'
            useradd -mr -g "$WG_ADMIN_GROUP" -p "$password" "$WG_SUPPORT_USER"

            # Force $WG_SUPPORT_USER to set new password at first login
            chage -d 0 "$WG_SUPPORT_USER"

        elif [ $rc -eq 0 ] ; then

            # Make sure $WG_SUPPORT_USER system account has
            # $WG_ADMIN_GROUP as the primary group.
            echo "Setting $WG_SUPPORT_USER system account primary group to $WG_ADMIN_GROUP system group"
            usermod -g "$WG_ADMIN_GROUP" "$WG_SUPPORT_USER" 2>&1 | grep -v "usermod: no changes" || true

        fi

        # Check for existence of group adm
        if getent group adm >/dev/null ; then

            # Add $WG_SUPPORT_USER system account to adm group
            echo "Adding $WG_SUPPORT_USER system account to adm system group"
            usermod -a -G adm "$WG_SUPPORT_USER"

            # Work-around (hack) for bug in upstart
            # (https://bugs.launchpad.net/ubuntu/+source/upstart/+bug/1074564)

            # Check for directory /var/log/upstart with gid adm
            if [ -d /var/log/upstart ] ; then
                if [ "`ls -ld /var/log/upstart | awk '{print $4}'`" != "adm" ] ; then
                    echo "Fixing directory /var/log/upstart ownership/permissions"
                    chgrp -R adm /var/log/upstart
                    chmod g+s /var/log/upstart
                fi
            fi

        fi

        # Check for existence of group fuse
        if getent group fuse >/dev/null ; then
            echo "Adding $WG_ADMIN_USER system account to fuse system group"
            usermod -a -G fuse "$WG_ADMIN_USER" 2>&1 | grep -v "usermod: no changes" || true
            echo "Adding $WG_SUPPORT_USER system account to fuse system group"
            usermod -a -G fuse "$WG_SUPPORT_USER" 2>&1 | grep -v "usermod: no changes" || true
        fi

        # Copy the pam_apparmor profiles and restart apparmor.
        if [ -d "${WG_SHAREDIR}/wg_system/apparmor-profiles" ] ; then
            if [ "${1}" = "install" ] ; then
                echo "Setting up pam_apparmor profiles"
            else
                echo "Updating pam_apparmor profiles"
            fi

            # Create our variables file
            mkdir -p /etc/apparmor.d/tunables
            /bin/rm -f /etc/apparmor.d/tunables/wg_system >/dev/null || true
            cat > /etc/apparmor.d/tunables/wg_system << CTRL_EOF
@{WG_ETCDIR}=${WG_ETCDIR}/
@{WG_BINDIR}=${WG_BINDIR}/
@{WG_LIBDIR}=${WG_LIBDIR}/
CTRL_EOF
            chmod 0644 /etc/apparmor.d/tunables/wg_system 

            if [ -d "${WG_SHAREDIR}/wg_system/apparmor-profiles/pam" ] ; then
                mkdir -p /etc/apparmor.d/pam
                cp -f ${WG_SHAREDIR}/wg_system/apparmor-profiles/pam/* /etc/apparmor.d/pam
            fi
            cp -f ${WG_SHAREDIR}/wg_system/apparmor-profiles/pam_* /etc/apparmor.d

            # Get AppArmor to reload profiles.
            if [ -x /etc/init.d/apparmor ] ; then
                /etc/init.d/apparmor restart || true
            fi
        fi

        # Copy the pam profiles.
        if [ "${1}" = "install" ] ; then
            echo "Setting up pam profiles"
        else
            echo "Updating pam profiles"
        fi

	for file in /etc/pam.d/login /etc/pam.d/sshd ; do
	    tmpfile=`mktemp`

            # The following sed command is a work-around for an issue
            # encountered in Ubuntu 14.04 LTS testing causing sshd
            # logins to fail and resulting in /var/log/auth.log
            # messages such as "error: PAM: pam_open_session(): Cannot
            # make/remove an entry for the specified session" and
            # "System error"
            sed -e 's/^\(session    required     pam_loginuid.so\)$/\# \1/' $file >$tmpfile

            if ! grep -q pam_apparmor.so $tmpfile ; then
                cat <<EOF >>$tmpfile

session    optional     pam_apparmor.so order=user,group,default
EOF
            fi

            if /usr/bin/cmp -s $file $tmpfile ; then
                /bin/rm -f $tmpfile
            else
                /bin/mv -f $tmpfile $file
                chmod 644 $file
            fi
        done

        # Setup the motd update 
        if [ -x "${WG_BINDIR}/wg_system_motd.sh" ] ; then
            WG_SYSTEM_UPDATE_MOTD=/etc/update-motd.d/50-wg-system
            /bin/rm -f ${WG_SYSTEM_UPDATE_MOTD} >/dev/null || true
            ln -sf "${WG_BINDIR}/wg_system_motd.sh" ${WG_SYSTEM_UPDATE_MOTD}
        fi

        # Copy the upstart init scripts
        cp -fp ${WG_SHAREDIR}/wg_system/init/wg_backup.conf /etc/init
        chown -R --reference=/etc/init /etc/init/wg_backup.conf
        cp -fp ${WG_SHAREDIR}/wg_system/init/wg_callhome.conf /etc/init
        chown -R --reference=/etc/init /etc/init/wg_callhome.conf
        cp -fp ${WG_SHAREDIR}/wg_system/init/wg_event.conf /etc/init
        chown -R --reference=/etc/init /etc/init/wg_event.conf
        cp -fp ${WG_SHAREDIR}/wg_system/init/wg_systemd.conf /etc/init
        chown -R --reference=/etc/init /etc/init/wg_systemd.conf
        cp -fp ${WG_SHAREDIR}/wg_system/init/wg_system_init.conf /etc/init
        chown -R --reference=/etc/init /etc/init/wg_system_init.conf
        cp -fp ${WG_SHAREDIR}/wg_system/init/wg_system_reset.conf /etc/init
        chown -R --reference=/etc/init /etc/init/wg_system_reset.conf
        cp -fp ${WG_SHAREDIR}/wg_system/init/wg_vm_detect.conf /etc/init
        chown -R --reference=/etc/init /etc/init/wg_vm_detect.conf
        cp -fp ${WG_SHAREDIR}/wg_system/init/wg_apttoken.conf /etc/init
        chown -R --reference=/etc/init /etc/init/wg_apttoken.conf

        # Copy the dhclient hook scripts
        if [ -d /etc/dhcp/dhclient-enter-hooks.d ] ; then
            cp ${WG_SHAREDIR}/wg_system/dhclient/wg_dhcp_enter_hook /etc/dhcp/dhclient-enter-hooks.d
        fi
        if [ -d /etc/dhcp/dhclient-exit-hooks.d ] ; then
            cp ${WG_SHAREDIR}/wg_system/dhclient/wg_dhcp_exit_hook /etc/dhcp/dhclient-exit-hooks.d
        fi

        # Copy the postinst.d files
        if [ -d /etc/kernel/postinst.d ]; then
            cp ${WG_SHAREDIR}/wg_system/kernel/postinst.d/zz-extlinux /etc/kernel/postinst.d/
        fi

        # Copy the default configuration files.
        mkdir -p ${WG_ETCDIR}/system/defaults
        cp -fp ${WG_SHAREDIR}/wg_system/defaults/*.json ${WG_ETCDIR}/system/defaults
        chmod 664 ${WG_ETCDIR}/system/defaults/*.json

        if [ "${1}" = "upgrade" ] ; then
            if [ ! -f ${WG_ETCDIR}/system/system.json ] ; then
            # Upgrading a system in factory-default state.
                cp -fp ${WG_ETCDIR}/system/defaults/* ${WG_ETCDIR}/system
            else
            # Copy new/missing configuration files from default directory
                for name in $(find ${WG_ETCDIR}/system/defaults/ -type f -printf "%f ")
                do
                    if [ ! -f ${WG_ETCDIR}/system/$name ] ; then
                        cp -fp ${WG_ETCDIR}/system/defaults/$name ${WG_ETCDIR}/system
                    fi
                done
            fi
        fi

        chown -R --reference=${WG_ETCDIR} ${WG_ETCDIR}/system

        # Create backup config directory and mount point.
        if [ ! -d ${WG_ETCDIR}/backup ] ; then
            mkdir -p ${WG_ETCDIR}/backup
            chown -R --reference=${WG_ETCDIR} ${WG_ETCDIR}/backup
            chmod 775 ${WG_ETCDIR}/backup
        fi

        if /bin/mount | grep -q "${WG_VARDIR}/backup_mnt\>" ; then
            # Currently mounted on, so it exists.
            :
        elif [ ! -d ${WG_VARDIR}/backup_mnt ] ; then
            mkdir -p ${WG_VARDIR}/backup_mnt
            chown -R --reference=${WG_VARDIR} ${WG_VARDIR}/backup_mnt
            chmod 775 ${WG_VARDIR}/backup_mnt
        fi

        # Install/update sshd trusted CA key
        cp -fp ${WG_SHAREDIR}/wg_system/defaults/sshd_ca_key.pub /etc/ssh/sshd_ca_keys
        chown --reference=/etc/ssh /etc/ssh/sshd_ca_keys
        chmod 644 /etc/ssh/sshd_ca_keys

        # Make sure wgadmin is in the sudoers list.
        if [ -f /etc/sudoers.d/wg_${WG_PRODUCT} ] ; then
            mv /etc/sudoers.d/wg_${WG_PRODUCT} /etc/sudoers.d/wg_${WG_PRODUCT}~
        fi
        tmpf=`mktemp`
        cat <<EOF >"$tmpf"
# wgadmin user is the default user in ${WG_PRODUCT} images.
# This user needs passwordless sudo functionality.
#
# NOTE: THIS FILE IS AUTOMATICALLY GENERATED.
#    ALL EDITS TO THIS FILE WILL BE LOST ON ${WG_PRODUCT} UPGRADES.
${WG_ADMIN_USER} ALL=(ALL) NOPASSWD:ALL
EOF
        /bin/mv -f "$tmpf" /etc/sudoers.d/wg_${WG_PRODUCT}
        chmod 440 /etc/sudoers.d/wg_${WG_PRODUCT}

        # Create the task directories
        mkdir -p ${WG_VARDIR}/system/task-requests
        mkdir -p ${WG_VARDIR}/system/task-status
        chown -R --reference=${WG_VARDIR} ${WG_VARDIR}/system

        # Create the upload directory
        mkdir -p ${WG_VARDIR}/upload
        chown -R --reference=${WG_VARDIR} ${WG_VARDIR}/upload

        # Create the tmp directory,
        if [ -d ${WG_VARDIR}/data ] ; then
            mkdir -p ${WG_VARDIR}/data/tmp
            chown -R --reference=${WG_VARDIR} ${WG_VARDIR}/data/tmp
        fi

        # Install daily cron job to retrieve APT token and remove expired
        # files in $WG_VARDIR
        tmpf=`mktemp`
        cat <<EOF >"$tmpf"
#!/bin/sh
# ${WG_PRODUCT} daily cron job
#
# NOTE: THIS FILE IS AUTOMATICALLY GENERATED.
#    ALL EDITS TO THIS FILE WILL BE LOST ON ${WG_PRODUCT} UPGRADES.

DEFAULT=/etc/default/wg_system

# read our config or else quit
test -r "\$DEFAULT" || exit 0
. "\$DEFAULT" || exit 1

# clear out local repository of retrieved package files
apt-get clean

# Check and, if needed, try to get an APT access token
/sbin/initctl start --no-wait wg_apttoken || true

# change to \$WG_VARDIR directory
cd "\${WG_VARDIR:?}" || exit 1

# find and remove log files modified 30 days ago or prior
find logs -type f -mtime +29 -print0 | xargs -0 /bin/rm -f

# find and remove system task request files modified 30 days ago or prior
find system/task-requests -type f -mtime +29 -print0 | xargs -0 /bin/rm -f

# find and remove system task status files modified 30 days ago or prior
find system/task-status -type f -mtime +29 -print0 | xargs -0 /bin/rm -f

# find and remove upload files modified one day ago or prior
find upload -type f -mtime +0 -print0 | xargs -0 /bin/rm -f

# find and remove temporary files modified one day ago or prior
if [ -d data/tmp ] ; then
    find data/tmp -type f -mtime +0 -print0 | xargs -0 /bin/rm -f
fi
EOF
        /bin/mv -f "$tmpf" /etc/cron.daily/wg_${WG_PRODUCT}
        chmod 755 /etc/cron.daily/wg_${WG_PRODUCT}

        mkdir -p "${WG_APTDIR}"

        if [ "${1}" = "upgrade" ] ; then

            # Start wg_systemd, but only on upgrades.
            echo "Starting wg_systemd"
            /sbin/initctl start wg_systemd >/dev/null || true

            # Start the wg_backup upstart process. Don't wait for it to
            # start though since this could potentially take a long time.
            echo "Starting wg_backup"
            /sbin/initctl start --no-wait wg_backup >/dev/null || true

            # Start the callhome service
            echo "Starting wg_callhome"
            /sbin/initctl start wg_callhome >/dev/null || true

            # Try to retrieve an APT token immediately on upgrades. If this
            # fails the daily cronjob will keep trying.
            echo "Starting wg_apttoken"
            /sbin/initctl start --no-wait wg_apttoken >/dev/null || true

            # Redetect virtualization platform
            echo "Starting wg_vm_detect"
            /sbin/initctl start --no-wait wg_vm_detect >/dev/null || true

            # Clean up any local log archives accidentally enabled by 1.0
            # users. These just consume space and are not accessible to
            # users.
            /bin/rm -f ${WG_VARDIR}/wlogserver/backup/*.zip

            # Cleanup APT archives
            apt-get clean || true

            # Update APT configuration and keyring
            update_apt_keyring

            # Upgrade /etc/fstab, setting <pass> to 1 or 2, depending upon the
            # file mount point.  A <mount point> of "/" sets <pass> to 1, a
            # <mount point> "/.+" sets <pass> to 2.

            tmpf=`mktemp`
            awk -f - /etc/fstab <<EOF >"$tmpf"
BEGIN {
    FS = "[ \t]+";
    format = "%-21s %-35s %-7s %-24s %-7s %-7s\n";
}
/^# <file system>/ {
    printf(format, "# <file system>", "<mount point>", "<type>", "<options>", "<dump>", "<pass>");
    next;
}
/^#/ {
    print \$0;
    next;
}
/^$/ {
    print \$0;
    next;
}
{
    pass = 0;
    if ( match(\$3, "^ext[234]\$") )
        if ( \$2 == "/" )
            pass = 1;
        else if ( match(\$2, "^/[^/]+") )
            pass = 2;
    printf(format, \$1, \$2, \$3, \$4, \$5, pass);
}
EOF
            /bin/mv -f "$tmpf" /etc/fstab
            chmod 644 /etc/fstab

            # Upgrade /etc/default/rcS, setting FSCKFIX=yes.

            tmpf=`mktemp`
            sed -e 's/^FSCKFIX=no$/FSCKFIX=yes/' /etc/default/rcS >"$tmpf"
            /bin/mv -f "$tmpf" /etc/default/rcS
            chmod 644 /etc/default/rcS

        fi

        ;;

    "uninstall")

        /bin/rm -f /etc/cron.daily/wg_${WG_PRODUCT}

        if [ -f /etc/init/wg_systemd.conf ] ; then
            echo "Stopping wg_systemd"
            /sbin/initctl stop wg_systemd >/dev/null || true
            /bin/rm -f /etc/init/wg_systemd.conf
        fi

        if [ -f /etc/init/wg_backup.conf ] ; then
            echo "Stopping wg_backup"
            /sbin/initctl stop wg_backup >/dev/null || true
            /bin/rm -f /etc/init/wg_backup.conf
        fi

        # remove callhome CRON job
        if [ -f /var/spool/cron/crontabs/${WG_ADMIN_USER} ] ; then
            /bin/rm -f /var/spool/cron/crontabs/${WG_ADMIN_USER}
        fi

        # remove upstart profiles
        /bin/rm -f /etc/init/wg_apttoken.conf
        /bin/rm -f /etc/init/wg_callhome.conf
        /bin/rm -f /etc/init/wg_event.conf
        /bin/rm -f /etc/init/wg_system_init.conf
        /bin/rm -f /etc/init/wg_system_reset.conf
        /bin/rm -f /etc/init/wg_vm_detect.conf

        /bin/rm -f /etc/update-motd.d/50-wg-system

        ;;

    *)

        usage $0

        ;;

esac

exit 0
